package WWW::Hetzner::Robot::API::Keys;
our $VERSION = '0.001';
our $AUTHORITY = 'cpan:GETTY';

# ABSTRACT: Hetzner Robot SSH Keys API

use Moo;
use Carp qw(croak);
use WWW::Hetzner::Robot::Key;
use namespace::clean;

has client => (
    is       => 'ro',
    required => 1,
    weak_ref => 1,
);

sub _wrap {
    my ($self, $data) = @_;
    return WWW::Hetzner::Robot::Key->new(
        client => $self->client,
        %$data,
    );
}

sub _wrap_list {
    my ($self, $list) = @_;
    return [ map { $self->_wrap($_->{key}) } @$list ];
}

sub list {
    my ($self) = @_;
    my $result = $self->client->get('/key');
    return $self->_wrap_list($result // []);
}

sub get {
    my ($self, $fingerprint) = @_;
    croak "Fingerprint required" unless $fingerprint;
    my $result = $self->client->get("/key/$fingerprint");
    return $self->_wrap($result->{key});
}

sub create {
    my ($self, %params) = @_;
    croak "name required" unless $params{name};
    croak "data required" unless $params{data};

    my $result = $self->client->post('/key', {
        name => $params{name},
        data => $params{data},
    });
    return $self->_wrap($result->{key});
}

sub delete {
    my ($self, $fingerprint) = @_;
    croak "Fingerprint required" unless $fingerprint;
    return $self->client->delete("/key/$fingerprint");
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

WWW::Hetzner::Robot::API::Keys - Hetzner Robot SSH Keys API

=head1 VERSION

version 0.001

=head1 SYNOPSIS

    my $robot = WWW::Hetzner::Robot->new(...);

    # List all keys
    my $keys = $robot->keys->list;

    # Get specific key
    my $key = $robot->keys->get('aa:bb:cc:...');

    # Create new key
    my $key = $robot->keys->create(
        name => 'my-key',
        data => 'ssh-ed25519 AAAA...',
    );

    # Delete key
    $robot->keys->delete('aa:bb:cc:...');

=head1 NAME

WWW::Hetzner::Robot::API::Keys - Hetzner Robot SSH Keys API

=head1 METHODS

=head2 list

Returns arrayref of L<WWW::Hetzner::Robot::Key> objects.

=head2 get

    my $key = $robot->keys->get($fingerprint);

Returns L<WWW::Hetzner::Robot::Key> object.

=head2 create

    my $key = $robot->keys->create(name => 'my-key', data => 'ssh-ed25519 ...');

Creates key and returns L<WWW::Hetzner::Robot::Key> object.

=head2 delete

    $robot->keys->delete($fingerprint);

=for :stopwords cpan testmatrix url bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/Getty/p5-www-hetzner>

  git clone https://github.com/Getty/p5-www-hetzner.git

=head1 AUTHOR

Torsten Raudssus <torsten@raudssus.de>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2026 by Torsten Raudssus.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
